from flask import Flask, render_template, request, send_file, make_response
from io import BytesIO
from pathlib import Path
from reportlab.pdfgen import canvas
from reportlab.lib.units import mm
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont
from PyPDF2 import PdfReader, PdfWriter
import os, json

# Try to register DejaVu font for better compatibility
try:
    pdfmetrics.registerFont(TTFont('DejaVu', '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf'))
    FONT = 'DejaVu'
except Exception:
    FONT = 'Helvetica'

BASE = Path(__file__).parent
TEMPLATE = BASE / "template.pdf"
CONFIG = {
  "positions": {
    "titulaire": {"x_mm": 20, "y_mm": 210, "font_size": 12},
    "iban": {"x_mm": 140, "y_mm": 235, "font_size": 11, "prefix": ""},
    "bic": {"x_mm": 140, "y_mm": 222, "font_size": 11, "prefix": ""},
    "adresse": {"x_mm": 20, "y_mm": 195, "font_size": 11}
  }
}

app = Flask(__name__)

def mm(v): return v * mm_unit
# we set mm_unit at runtime because reportlab mm is not JSON-serializable here

@app.get("/")
def index():
    defaults = {
        "titulaire": "P.e. Fonkoua kenne",
        "iban": "FR76 2763 3121 2905 0404 4032 833",
        "bic": "BUNQFRP2",
        "adresse": "d'Alsace 28 94550 Chevilly-Larue\nFrance"
    }
    return render_template("admin.html", defaults=defaults)

@app.post("/generate")
def generate():
    # read form
    titulaire = request.form.get("titulaire","").strip()
    iban = request.form.get("iban","").strip()
    bic = request.form.get("bic","").strip()
    adresse = request.form.get("adresse","").strip()

    # read template to get size
    reader = PdfReader(str(TEMPLATE))
    page = reader.pages[0]
    w = float(page.mediabox.width)
    h = float(page.mediabox.height)

    # create overlay in memory
    packet = BytesIO()
    c = canvas.Canvas(packet, pagesize=(w, h))
    # dynamic mm unit
    global mm_unit
    from reportlab.lib.units import mm as _mm
    mm_unit = _mm

    # Titulaire
    pos = CONFIG["positions"]["titulaire"]
    c.setFont(FONT, pos.get("font_size", 12))
    c.drawString(pos["x_mm"]*mm_unit, pos["y_mm"]*mm_unit, titulaire)

    # IBAN
    pos = CONFIG["positions"]["iban"]
    c.setFont(FONT, pos.get("font_size", 11))
    pref = pos.get("prefix","")
    c.drawString(pos["x_mm"]*mm_unit, pos["y_mm"]*mm_unit, pref + iban)

    # BIC
    pos = CONFIG["positions"]["bic"]
    c.setFont(FONT, pos.get("font_size", 11))
    pref = pos.get("prefix","")
    c.drawString(pos["x_mm"]*mm_unit, pos["y_mm"]*mm_unit, pref + bic)

    # Adresse multi-line
    pos = CONFIG["positions"]["adresse"]
    c.setFont(FONT, pos.get("font_size", 11))
    x = pos["x_mm"]*mm_unit
    y = pos["y_mm"]*mm_unit
    for i, line in enumerate(adresse.splitlines() if adresse else []):
        c.drawString(x, y - i * 13, line)

    c.save()
    packet.seek(0)

    # merge overlay with original
    overlay_reader = PdfReader(packet)
    orig = reader
    page.merge_page(overlay_reader.pages[0])
    writer = PdfWriter()
    writer.add_page(page)
    out_stream = BytesIO()
    writer.write(out_stream)
    out_stream.seek(0)

    return send_file(out_stream, as_attachment=True, download_name="releve_bancaire.pdf", mimetype="application/pdf")

if __name__ == "__main__":
    # set mm_unit for module-level use
    from reportlab.lib.units import mm as mm_unit
    app.run(debug=True, host="0.0.0.0", port=5000)
